// ****************************************************************************
// GlottalImageExplorer.
// Copyright (C) 2015-2016 Peter Birkholz.
// This program is free and open-source software.
// ****************************************************************************

#include <wx/filename.h>
#include <wx/stdpaths.h>
#include <wx/busyinfo.h>
#include <iomanip>
#include <fstream>
#include <sstream>

#include "Data.h"
#include "ImageProc.h"


// ****************************************************************************
// Static data.
// ****************************************************************************

Data *Data::instance = NULL;


// ****************************************************************************
/// Returns the one instance of this class.
// ****************************************************************************

Data *Data::getInstance()
{
  if (instance == NULL)
  {
    instance = new Data();
  }
  return instance;
}

// ****************************************************************************
/// Init the data. This function must be called once after the first call of
/// getInstance().
// ****************************************************************************

void Data::init(char *arg0)
{
  int i;

  // ****************************************************************
  // Determine the program path from arg0. The option
  // wxPATH_GET_SEPARATOR makes sure that the path is always
  // terminated with a "\".
  // ****************************************************************

  wxFileName fileName(arg0);
  programPath = fileName.GetPath(wxPATH_GET_VOLUME | wxPATH_GET_SEPARATOR);
  printf("The program path is %s.\n", programPath.c_str());

  // ****************************************************************

  filmFileName = programPath;
  bmpFolderName = programPath;
  areaFileName = programPath;
  contourFileName = programPath;
  segmentationFileName = programPath;

  film = new Film();

  for (i=0; i < NUM_DISPLAYED_FRAMES; i++)
  {
    frameIndex[i] = 0;
  }

  selectionIndex[0] = 1000;
  selectionIndex[1] = 2000;
  imageType = COLOR_IMAGE;
  playbackSpeed_percent = 100;

  srgData = new SrgData[Film::MAX_FRAMES];
  resetSrgData();

  seedThresholdCoupling = true;
  showGlottisArea = false;
  showGlottisEdge = true;
}


// ****************************************************************************
// ****************************************************************************

void Data::resetSrgData()
{
  int i, k;

  for (i = 0; i < Film::MAX_FRAMES; i++)
  {
    srgData[i].isUserDefinedSegmentation = false;
    
    for (k = 0; k < NUM_SEEDS; k++)
    {
      srgData[i].seed[k].x = Film::WIDTH / 2;
      srgData[i].seed[k].y = (k + 1) * Film::HEIGHT / (NUM_THRESHOLD_POINTS + 1);
    }

    for (k = 0; k < NUM_THRESHOLD_POINTS; k++)
    {
      srgData[i].thresholdY[k] = srgData[i].seed[k].y;
      srgData[i].thresholdValue[k] = 80;
    }

    srgData[i].glottisArea_pix = 0;
  }

}


// ****************************************************************************
/// Returns the current frame index or -1, if the current frame is invalid.
// ****************************************************************************

int Data::getCurrFrameIndex(int pictureIndex)
{
  int numFrames = film->getNumFrames();
  int frame = frameIndex[pictureIndex];

  if (frame < 0)
  {
    frame = -1;
  }
  if (frame >= numFrames)
  {
    frame = -1;
  }

  return frame;
}


// ****************************************************************************
/// Interpolates the SRG data in time.
// ****************************************************************************

void Data::interpolateSrgData()
{
  // Make a list of frames with user defined SRG data.
  vector<int> userFrames;
  int i, k, m;

  int numFrames = film->getNumFrames();
  if (numFrames < 1)
  {
    return;
  }

  for (i = 0; i < numFrames; i++)
  {
    if (srgData[i].isUserDefinedSegmentation)
    {
      userFrames.push_back(i);
    }
  }

  int numUserFrames = (int)userFrames.size();

  if (numUserFrames < 1)
  {
    resetSrgData();
    return;
  }

  // ****************************************************************
  // All frames up to the first user frame
  // ****************************************************************

  for (i = 0; i < userFrames[0]; i++)
  {
    srgData[i] = srgData[userFrames[0]];
    srgData[i].isUserDefinedSegmentation = false;
  }


  // ****************************************************************
  // All frames between the first and last user frame
  // ****************************************************************

  int frame1, frame2;
  double t, t1;

  for (k = 0; k < numUserFrames - 1; k++)
  {
    frame1 = userFrames[k];
    frame2 = userFrames[k + 1];

    for (i = frame1 + 1; i < frame2; i++)
    {
      t = (double)(i - frame1) / (double)(frame2 - frame1);   // 0 <= t <= 1
      t1 = 1.0 - t;

      for (m = 0; m < NUM_THRESHOLD_POINTS; m++)
      {
        srgData[i].thresholdY[m] = (int)(t1*srgData[frame1].thresholdY[m] + t*srgData[frame2].thresholdY[m]);
        srgData[i].thresholdValue[m] = (int)(t1*srgData[frame1].thresholdValue[m] + t*srgData[frame2].thresholdValue[m]);
      }

      for (m = 0; m < NUM_SEEDS; m++)
      {
        srgData[i].seed[m].x = (int)(t1*srgData[frame1].seed[m].x + t*srgData[frame2].seed[m].x);
        srgData[i].seed[m].y = (int)(t1*srgData[frame1].seed[m].y + t*srgData[frame2].seed[m].y);
      }
    }
  }

  // ****************************************************************
  // All frames after the last user frame
  // ****************************************************************

  for (i = userFrames[numUserFrames - 1] + 1; i < numFrames; i++)
  {
    srgData[i] = srgData[userFrames[numUserFrames - 1]];
    srgData[i].isUserDefinedSegmentation = false;
  }
}


// ****************************************************************************
/// Load the user-defined segmentation data.
// ****************************************************************************

bool Data::loadSegmentationData(const char *fileName)
{
  if (fileName == NULL)
  {
    return false;
  }

  ifstream is(fileName);
  int i, k;

  if (!is)
  {
    wxMessageBox(wxString("Could not open ") + wxString(fileName) + wxString(" for reading."),
      wxString("Error!"));
    return false;
  }

  // Reset the previous region growing data.
  resetSrgData();

  // ****************************************************************
  // Skip the header line.
  // ****************************************************************

  string header;
  getline(is, header);

  // ****************************************************************
  // Read the actual data.
  // ****************************************************************

  SrgData *d = NULL;
  string line;
  int index;
  bool yCoordinatesDiffer = false;

  while (getline(is, line))
  {
    istringstream ss(line);
    ss >> index;            // Index of the frame

    if ((index >= 0) && (index < Film::MAX_FRAMES))
    {
      d = &srgData[index];
      
      d->isUserDefinedSegmentation = true;
      
      for (k = 0; k < NUM_THRESHOLD_POINTS; k++)
      {
        ss >> d->thresholdY[k];
        ss >> d->thresholdValue[k];
      }

      for (k = 0; k < NUM_SEEDS; k++)
      {
        ss >> d->seed[k].x;
        ss >> d->seed[k].y;
      }

      // Check if y-coord. of seeds and threshold points differ.
      for (k = 0; k < NUM_THRESHOLD_POINTS; k++)
      {
        if (d->thresholdY[k] != d->seed[k].y)
        {
          yCoordinatesDiffer = true;
        }
      }
    }
    else
    {
      printf("Error: Invalid frame index!\n");
      is.close();
      resetSrgData();
      return false;
    }
  }

  // ****************************************************************
  // Potentially equalize the y-coordinates of seeds and thresholds.
  // ****************************************************************

  if ((seedThresholdCoupling) && (yCoordinatesDiffer))
  {
    wxMessageBox(
      "The y-coordinates of the threshold points will be adapted to the y-coordinates of the seeds!",
      "Warning");

    for (i = 0; i < Film::MAX_FRAMES; i++)
    {
      if (srgData[i].isUserDefinedSegmentation)
      {
        for (k = 0; k < NUM_SEEDS; k++)
        {
          srgData[i].thresholdY[k] = srgData[i].seed[k].y;
        }
      }
    }
  }

  // ****************************************************************

  interpolateSrgData();

  // ****************************************************************
  // Close the file.
  // ****************************************************************

  is.close();

  return true;
}


// ****************************************************************************
/// Save the user-defined segmentation data.
// ****************************************************************************

bool Data::saveSegmentationData(const char *fileName)
{
  if (fileName == NULL)
  {
    return false;
  }

  ofstream os(fileName);
  int i, k;

  if (!os)
  {
    wxMessageBox(wxString("Could not open ") + wxString(fileName) + wxString(" for writing."),
      wxString("Error!"));
    return false;
  }

  // ****************************************************************
  // Write the header.
  // ****************************************************************

  os << "# frame-index "
     << "threshold1-y threshold1-value "
     << "threshold2-y threshold2-value "
     << "threshold3-y threshold3-value "
     << "seed1-x seed1-y "
     << "seed2-x seed2-y "
     << "seed3-x seed3-y "
     << endl;

  // ****************************************************************
  // Write actual data.
  // ****************************************************************

  int numFrames = film->getNumFrames();
  SrgData *d = NULL;

  for (i = 0; i < numFrames; i++)
  {
    d = &srgData[i];
    if (d->isUserDefinedSegmentation)
    {
      os << i << " ";

      for (k = 0; k < NUM_THRESHOLD_POINTS; k++)
      {
        os << d->thresholdY[k] << " " << d->thresholdValue[k] << " ";
      }

      for (k = 0; k < NUM_SEEDS; k++)
      {
        os << d->seed[k].x << " " << d->seed[k].y << " ";
      }

      os << endl;
    }
  }

  // ****************************************************************
  // Close the file.
  // ****************************************************************

  os.close();

  return true;
}


// ****************************************************************************
/// Export the time function of the glottal area.
// ****************************************************************************

bool Data::exportGlottisArea(const char *fileName)
{
  if (fileName == NULL)
  {
    return false;
  }

  ofstream os(fileName);
  int i;

  if (!os)
  {
    wxMessageBox(wxString("Could not open ") + wxString(fileName) + wxString(" for writing."),
      wxString("Error!"));
    return false;
  }

  wxBusyInfo wait("Please wait while exporting the glottal area waveform...");

  // ****************************************************************
  // Write the actual data.
  // ****************************************************************

  int numFrames = film->getNumFrames();
  int numGlottisPixels = 0;
  int segmentMatrix[Film::WIDTH * Film::HEIGHT];
  int leftContour[Film::HEIGHT];
  int rightContour[Film::HEIGHT];

  for (i = 0; i < numFrames; i++)
  {
    numGlottisPixels = ImageProc::segmentGlottis(&film->getFrame(i), segmentMatrix, 
      leftContour, rightContour, &srgData[i]);
    os << numGlottisPixels << endl;
  }

  // ****************************************************************
  // Close the file.
  // ****************************************************************

  os.close();

  return true;
}


// ****************************************************************************
// Export the glottis contour waveform as a text file.
// ****************************************************************************

bool Data::exportGlottisContour(const char *fileName)
{
  if (fileName == NULL)
  {
    return false;
  }

  ofstream os(fileName);
  int i;

  if (!os)
  {
    wxMessageBox(wxString("Could not open ") + wxString(fileName) + wxString(" for writing."),
      wxString("Error!"));
    return false;
  }

  wxBusyInfo wait("Please wait while exporting the glottal contour waveform...");

  // ****************************************************************
  // Write the actual data.
  // ****************************************************************

  os << "# There are two lines of numbers for each image: they contain the left and right contour of the glottis, respectively." << endl;

  int y = 0;
  int numFrames = film->getNumFrames();
  int xLeft[Film::HEIGHT] = { 0 };
  int xRight[Film::HEIGHT] = { 0 };
  int segmentMatrix[Film::WIDTH * Film::HEIGHT];

  for (i = 0; i < numFrames; i++)
  {
    ImageProc::segmentGlottis(&film->getFrame(i), segmentMatrix, xLeft, xRight, &srgData[i]);

    // Write one line with the left border coordinates
    
    for (y = 0; y < Film::HEIGHT; y++)
    {
      os << xLeft[y] << " ";
    }
    os << endl;

    // Write one line with the right border coordinates

    for (y = 0; y < Film::HEIGHT; y++)
    {
      os << xRight[y] << " ";
    }
    os << endl;
  }

  // ****************************************************************
  // Close the file.
  // ****************************************************************

  os.close();

  return true;
}


// ****************************************************************************
/// Constructor.
// ****************************************************************************

Data::Data()
{
  // Do nothing. Initialization is done in init().
}

// ****************************************************************************
